<?php

namespace App\Http\Controllers;

use App\Models\IcaEnrollment;
use App\Models\IcaProgram;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;

class IcaEnrollmentController extends Controller
{
    public function index()
    {
        $programs = IcaProgram::active()->orderBy('category')->orderBy('program_name')->get();
        $programsByCategory = $programs->groupBy('category');
        
        return view('frontend.default.ica.enrollment_elegant', compact('programsByCategory'));
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'surname' => 'required|string|max:255',
            'first_name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'mobile' => 'required|string|max:20',
            'age' => 'required|integer|min:18|max:100',
            'address' => 'required|string',
            'city' => 'required|string|max:255',
            'educational_level' => 'required|in:SHS/TVET,DIPLOMA/DEGREE,MASTERS,OTHER',
            'programs_of_interest' => 'required|array|min:1',
            'programs_of_interest.*' => 'exists:ica_programs,program_code',
            'has_previous_experience' => 'required|boolean',
            'work_experience' => 'nullable|string',
            'payment_method' => 'required|in:paystack,bank_transfer,mobile_money',
            'bank_receipt' => 'required_if:payment_method,bank_transfer|file|mimes:jpg,jpeg,png,pdf|max:5120'
        ]);

        $enrollment = IcaEnrollment::create([
            'user_id' => auth()->id(),
            'surname' => $validated['surname'],
            'first_name' => $validated['first_name'],
            'email' => $validated['email'],
            'mobile' => $validated['mobile'],
            'age' => $validated['age'],
            'address' => $validated['address'],
            'city' => $validated['city'],
            'educational_level' => $validated['educational_level'],
            'enrollment_fee' => 100.00,
            'programs_of_interest' => $validated['programs_of_interest'],
            'has_previous_experience' => $validated['has_previous_experience'],
            'work_experience' => $validated['work_experience'] ?? 'No previous experience',
            'payment_method' => $validated['payment_method'],
            'payment_status' => 'pending',
            'enrollment_status' => 'pending'
        ]);

        if ($request->hasFile('bank_receipt')) {
            $path = $request->file('bank_receipt')->store('ica_receipts', 'public');
            $enrollment->bank_receipt_path = $path;
            $enrollment->save();
            
            return redirect()->route('ica.enrollment.success', ['id' => $enrollment->id])
                ->with('success', 'Expression of Interest submitted successfully! Your bank receipt has been uploaded for verification.');
        }

        if ($validated['payment_method'] === 'paystack') {
            return redirect()->route('ica.payment.paystack', ['enrollment_id' => $enrollment->id]);
        }

        return redirect()->route('ica.enrollment.success', ['id' => $enrollment->id])
            ->with('success', 'Expression of Interest submitted successfully!');
    }

    public function initiatePaystackPayment($enrollment_id)
    {
        $enrollment = IcaEnrollment::findOrFail($enrollment_id);
        
        $payment_gateway = DB::table('payment_gateways')->where('identifier', 'paystack')->first();
        
        if (!$payment_gateway || $payment_gateway->status != 1) {
            return redirect()->route('ica.enrollment')
                ->with('error', 'Paystack payment is currently unavailable. Please use bank transfer.');
        }

        $keys = json_decode($payment_gateway->keys, true);
        $test_mode = $payment_gateway->test_mode == 1;
        $public_key = $test_mode ? $keys['public_test_key'] : $keys['public_live_key'];

        $page_data = [
            'enrollment' => $enrollment,
            'public_key' => $public_key,
            'amount' => $enrollment->enrollment_fee * 100,
            'email' => $enrollment->email,
            'reference' => 'ICA-' . time() . '-' . $enrollment->id
        ];

        return view('frontend.default.ica.paystack_payment', $page_data);
    }

    public function verifyPaystackPayment(Request $request)
    {
        $reference = $request->input('reference');
        $enrollment_id = $request->input('enrollment_id');
        
        $enrollment = IcaEnrollment::findOrFail($enrollment_id);
        
        $payment_gateway = DB::table('payment_gateways')->where('identifier', 'paystack')->first();
        $keys = json_decode($payment_gateway->keys, true);
        $test_mode = $payment_gateway->test_mode == 1;
        $secret_key = $test_mode ? $keys['secret_test_key'] : $keys['secret_live_key'];

        $curl = curl_init();
        curl_setopt_array($curl, array(
            CURLOPT_URL => "https://api.paystack.co/transaction/verify/" . $reference,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_ENCODING => "",
            CURLOPT_MAXREDIRS => 10,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_CUSTOMREQUEST => "GET",
            CURLOPT_HTTPHEADER => array(
                "Authorization: Bearer " . $secret_key,
                "Cache-Control: no-cache",
            ),
        ));

        $response = curl_exec($curl);
        $err = curl_error($curl);
        curl_close($curl);

        if ($err) {
            return redirect()->route('ica.enrollment')
                ->with('error', 'Payment verification failed. Please try again.');
        }

        $result = json_decode($response, true);

        if ($result['status'] && $result['data']['status'] === 'success') {
            $enrollment->payment_status = 'paid';
            $enrollment->payment_reference = $reference;
            $enrollment->payment_verified_at = now();
            $enrollment->save();

            return redirect()->route('ica.enrollment.success', ['id' => $enrollment->id])
                ->with('success', 'Payment successful! Your enrollment has been confirmed.');
        }

        return redirect()->route('ica.enrollment')
            ->with('error', 'Payment verification failed. Please contact support.');
    }

    public function success($id)
    {
        $enrollment = IcaEnrollment::with('programs')->findOrFail($id);
        return view('frontend.default.ica.success', compact('enrollment'));
    }

    public function scaffoldingProgram()
    {
        return view('frontend.default.ica.scaffolding_program');
    }
}
